/* Lachesis - functions for file descriptor based i/o
 * Copyright 2003-2004 M. Dennis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include "alloc.h"
#include "utils.h"

static char* id="@(#) $Id$";

ssize_t fdgetline(char **input, size_t *len, int fd)
{
  int c, r, l;

  if (*len==0) {
    *len = 32;
    *input = (char *)malloc(32);
  }
  l=32;
  for (c=0;1;c++) {
    if (c>=*len+1) {
      char *temp;
      if (l==32)
	while (l<=*len)
	  l*=2;
      else
	l*=2;
      temp = (char *)realloc(*input, l);
      if (temp == NULL) {
	Util_Eprintf("Memory allocation error: %m\n");
	exit (255);
      }
      *input = temp;
      *len = l;
    }
    r = read(fd, *input+c, 1);
    if (r==-1) {
      if (errno!=EAGAIN) {
	Util_Eprintf("fdgetline: read: %m\n");
	return -1;
      } else {
	(*input)[c]='\n';
	(*input)[c+1]=0;
	Util_Eprintf("fdinput: read would block! Input so far: %s", 
		     *input);
	return c+1;
      }
    } else if (r==0)
      return 0;
    if ((*input)[c]=='\n') {
      (*input)[c+1] = 0;
      return c+1;
    }
  }
}

int fdputs(const char *S, int fd)
{
  // This function will perform blocking writes even for non-blocking
  // descriptors.
  fd_set writes;
  int c;
  ssize_t r;
  const char *tail;

  c = 0;
  tail = strchr(S, 0);
  while (S[c]) {
    FD_ZERO(&writes);
    FD_SET(fd, &writes);
    select(FD_SETSIZE, NULL, &writes, NULL, NULL);
    r = write(fd, S+c, tail-(S+c));
    if (r==-1) {
      if (errno!=EAGAIN)
	return EOF;
    } else
      c+=r;
  }
  return 0;
}

int fdprintf(int fd, const char *template, ...)
{
  va_list ap;
  int r;
  char *buffer;
  va_start(ap, template);
  r = vasprintf(&buffer, template, ap);
  va_end(ap);
  if (r<0)
    return r;
  if (fdputs(buffer, fd)<0)
    r = -1;
  free (buffer);
  return r;
}
