/*
 * Lachesis, an IRCRPG combat engine - Dice roller interface
 * Copyright 2003-2004 M. Dennis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Note: This is a standalone dice roll for user benefit.
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "root.h"
#include "roll.h"
#include "../engine/random.h"
#include "../auth.h"
#include "../utils.h"
#include "../lachesis.h"
#include "../ircint.h"
#include "../config.h"

static char* id="@(#) $Id$";

extern CmdOpts options;
extern char* irc_curchan;

#ifndef RPGSERV
extern Cmd_PMod cmd_modules;

void Cmd_roll_Init()
{
  Cmd_Alloc(&Cmd_Roll_Root, "roll", CMD_AUTH_USER);
  Cmd_Alloc(&Cmd_WRoll_Root, "wroll", CMD_AUTH_USER);
}
#endif

void Cmd_Roll_Root(void *ticket, int auth, const char *str)
{
  unsigned long count, magnitude;
  long floor;
  char tstr[32];
  const char *temp, *ptr = str;
#ifdef RPGSERV
#ifndef ALLOW_ANY_TO_ROLL
  if (auth<CMD_AUTH_USER) {
    Auth_CmdResult(ticket, auth, 
		   "You are not authorized to use this command.");
    return ;
  }
#endif
#endif
  if (str==NULL||!*str)
    Auth_CmdResult(ticket, auth, "Usage: roll [#]d#[{+#|-#}]");
  else {
    Util_SepTo(tstr, 32, &ptr, 'd');
    if (ptr==NULL) {
      Auth_CmdResult(ticket, auth, "Unrecognized roll format.");
      return ;
    }
    if (*tstr) {
      if (*tstr<'0'||*tstr>'9') {
	Auth_CmdResult(ticket, auth, "Unrecognized roll format.");
	return ;
      } else
	count = strtoul(tstr, NULL, 10);
    } else
      count = 1;
    temp = strchr(ptr, '+');
    if (temp==NULL)
      temp = strchr(ptr, '-');
    if (temp==NULL) {
      if (ptr[0]<'0'||ptr[0]>'9') {
	Auth_CmdResult(ticket, auth, "Unrecognized roll format.");
	return ;
      }
      magnitude = strtoul(ptr, NULL, 10);
      Cmd_Roll_DoRoll(ticket, auth, count, magnitude, 1);
    } else {
      Util_SepTo(tstr, 32, &ptr, *temp);
      if (tstr[0]<'0'||tstr[0]>'9'||ptr[0]<'0'||ptr[0]>'9') {
	Auth_CmdResult(ticket, auth, "Unrecognized roll format.");
	return ;
      }
      magnitude = strtoul(tstr, NULL, 10);
      floor = strtol(ptr-1, NULL, 10);
      Cmd_Roll_DoRoll(ticket, auth, count, magnitude, 1+floor);
    }
  }
}

void Cmd_WRoll_Root(void *ticket, int auth, const char *str)
{
  unsigned long count, threshold;
  char tstr[32];
  const char *temp, *ptr = str;
#ifdef RPGSERV
#ifndef ALLOW_ANY_TO_ROLL
  if (auth<CMD_AUTH_USER) {
    Auth_CmdResult(ticket, auth, 
		   "You are not authorized to use this command.");
    return ;
  }
#endif
#endif
  if (str==NULL||!*str)
    Auth_CmdResult(ticket, auth, "Usage: wroll #-#");
  else {
    Util_SepTo(tstr, 32, &ptr, '-');
    if (ptr==NULL) {
      Auth_CmdResult(ticket, auth, "Unrecognized wroll format.");
      return ;
    }
    if (*tstr) {
      if (*tstr<'0'||*tstr>'9') {
	Auth_CmdResult(ticket, auth, "Unrecognized wroll format.");
	return ;
      } else
	count = strtoul(tstr, NULL, 10);
    } else
      count = 1;
    if (ptr[0]<'0'||ptr[0]>'9') {
      Auth_CmdResult(ticket, auth, "Unrecognized roll format.");
      return ;
    }
    threshold = strtoul(ptr, NULL, 10);
    Cmd_WRoll_DoRoll(ticket, auth, count, threshold);
  }
}

void Cmd_Roll_DoRoll(void *ticket, int auth, unsigned long count, 
		     unsigned long magnitude, long floor)
{
  unsigned long loop, temp;
  long *rolls;
  char **lfmt;

  if (count==0||count>DICE_MAX) {
    Auth_CmdResult(ticket, auth, "Invalid count.");
    return ;
  }
  if (magnitude<2) {
    Auth_CmdResult(ticket, auth, "Invalid magnitude.");
    return ;
  }
  if (!Random_Roll(count, magnitude, floor, &rolls)) {
    Auth_CmdResult(ticket, auth, "Error performing dice roll.");
    return ;
  }

  switch(count) {
  case 1:
    Cmd_Roll_Print(ticket, auth, Util_Format("roll is %d", rolls[0]));
    break;
  case 2:
    Cmd_Roll_Print(ticket, auth, Util_Format("roll is %d: %d and %d",
					     rolls[0], rolls[1], rolls[2]));
    break;
  default:
    lfmt = new char*[count-1];
    temp = 0;
    for (loop=1;loop<count-1;loop++) {
      asprintf(lfmt+loop, "%d, ", rolls[loop]);
      temp += strlen(lfmt[loop]);
    }
    lfmt[0] = new char[temp+1];
    lfmt[0][0] = '\0';
    for (loop=1;loop<count-1;loop++) {
      strcat(lfmt[0], lfmt[loop]);
      free(lfmt[loop]);
    }
    Cmd_Roll_Print(ticket, auth, Util_Format("roll is %d: %s%d and %d",
					     rolls[0], lfmt[0], 
					     rolls[count-1], rolls[count])
		   );
    delete [] lfmt[0];
    delete [] lfmt;
  }
}

void Cmd_WRoll_DoRoll(void *ticket, int auth, unsigned long count, 
		      unsigned long threshold)
{
  unsigned long loop, temp;
  long *rolls;
  char **lfmt;

  if (count==0||count>DICE_MAX) {
    Auth_CmdResult(ticket, auth, "Invalid count.");
    return ;
  }
  if (threshold<2||threshold>10) {
    Auth_CmdResult(ticket, auth, "Invalid magnitude.");
    return ;
  }
  if (!Random_WRoll(count, threshold, &rolls)) {
    Auth_CmdResult(ticket, auth, "Error performing dice roll.");
    return ;
  }

  switch(count) {
  case 1:
    if (rolls[0]>0)
      Cmd_WRoll_Print(ticket, auth, Util_Format("succeeded: %d", rolls[1]));
    else if (rolls[0]==0)
      Cmd_WRoll_Print(ticket, auth, Util_Format("failed: %d", rolls[1]));
    else
      Cmd_WRoll_Print(ticket, auth, "botched it!");
    break;
  case 2:
    if (rolls[0]>1)
      Cmd_WRoll_Print(ticket, auth, Util_Format("had %d successes: %d and %d",
						rolls[0], rolls[1], rolls[2]));
    else if (rolls[0]==1)
      Cmd_WRoll_Print(ticket, auth, 
		      Util_Format("succeeded once: %d and %d", rolls[1],
				  rolls[2])
		      );
    else if (rolls[0]==0)
      Cmd_WRoll_Print(ticket, auth, Util_Format("failed: %d and %d", rolls[1],
						rolls[2]));
    else
      Cmd_WRoll_Print(ticket, auth, Util_Format("botched it: %d and %d",
						rolls[1], rolls[2]));
    break;
  default:
    lfmt = new char*[count-1];
    temp = 0;
    for (loop=1;loop<count-1;loop++) {
      asprintf(lfmt+loop, "%d, ", rolls[loop]);
      temp += strlen(lfmt[loop]);
    }
    lfmt[0] = new char[temp+1];
    lfmt[0][0] = '\0';
    for (loop=1;loop<count-1;loop++) {
      strcat(lfmt[0], lfmt[loop]);
      free(lfmt[loop]);
    }
    if (rolls[0]>1)
      Cmd_WRoll_Print(ticket, auth, 
		      Util_Format("had %d successes: %s%d and %d", rolls[0], 
				  lfmt[0], rolls[count-1], rolls[count])
		      );
    else if (rolls[0]==1)
      Cmd_WRoll_Print(ticket, auth,
		      Util_Format("succeeded once: %s%d and %d", lfmt[0],
				  rolls[count-1], rolls[count])
		      );
    else if (rolls[0]==0)
      Cmd_WRoll_Print(ticket, auth, Util_Format("failed: %s%d and %d", lfmt[0],
						rolls[count-1], rolls[count])
		      );
    else
      Cmd_WRoll_Print(ticket, auth, Util_Format("botched it: %s%d and %d",
						lfmt[0], rolls[count-1],
						rolls[count])
		      );
    delete [] lfmt[0];
    delete [] lfmt;
  }
}

void Cmd_Roll_Print(void *ticket, int auth, const char *str)
{
  if (options.pubdice&&irc_curchan!=NULL&&*irc_curchan) {
    char *out;
    const char *nick;
    switch (auth) {
    case CMD_AUTH_NONE:
      nick = ((Auth_PTicket)ticket)->GetNick();
      break;
    case CMD_AUTH_USER:
      nick = ((Auth_PUser)ticket)->GetNick();
      break;
    case CMD_AUTH_MASTER:
      nick = Auth_TUser::GetMaster()->GetNick();
      break;
    default:  // This should never happen.
      Util_Eprintf("Invalid auth value in Cmd_Roll_Print: %d", auth);
      return ;
    }
    out = new char[strlen(str)+strlen(nick)+9];
    sprintf(out, "OOC: %s's %s", nick, str);
    IRC_Notice(irc_curchan, out);
    delete [] out;
  }
  else {
    char *out;
    out = new char[strlen(str)+11];
    sprintf(out, "OOC: Your %s", str);
    Auth_CmdResult(ticket, auth, out);
    delete [] out;
  }
}

void Cmd_WRoll_Print(void *ticket, int auth, const char *str)
{
  if (options.pubdice&&irc_curchan!=NULL&&*irc_curchan) {
    char *out;
    const char *nick;
    switch (auth) {
    case CMD_AUTH_NONE:
      nick = ((Auth_PTicket)ticket)->GetNick();
      break;
    case CMD_AUTH_USER:
      nick = ((Auth_PUser)ticket)->GetNick();
      break;
    case CMD_AUTH_MASTER:
      nick = Auth_TUser::GetMaster()->GetNick();
      break;
    default:  // This should never happen.
      Util_Eprintf("Invalid auth value in Cmd_WRoll_Print: %d", auth);
      return ;
    }
    out = new char[strlen(str)+strlen(nick)+7];
    sprintf(out, "OOC: %s %s", nick, str);
    IRC_Notice(irc_curchan, out);
    delete [] out;
  }
  else {
    char *out;
    out = new char[strlen(str)+10];
    sprintf(out, "OOC: You %s", str);
    Auth_CmdResult(ticket, auth, out);
    delete [] out;
  }
}
