/*
 * Lachesis IRCRPG Combat Engine - Logging system
 * Copyright 2003-2004 M. Dennis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "lachesis.h"
#include "auth.h"
#include "types.h"
#include "alloc.h"
#include "log.h"
#include "utils.h"
#include "config.h"

static char* id="@(#) $Id: log.cc,v 1.1.1.1 2004/01/16 07:28:15 lachesis Exp $";

extern CmdOpts options;
FILE *log_file = NULL;
TF log_active = FALSE;
TF log_bquote = FALSE;

#define DFL_LOGTITLE "Final Fantasy: Dragoon's Birth"

enum {
  LOG_LINE,
  LOG_DESC,
  LOG_OOC,
  LOG_FIX,
};

void Log_Init()
{
  if (options.logfile==NULL||*options.logfile=='\0')
    return;
  log_file = fopen(options.logfile, "wx");
  if (log_file==NULL) {
    Util_Eprintf("Failed to create logfile %s: %m\n", options.logfile);
    return;
  }
  //  log_active = TRUE;
  log_bquote = FALSE;
  fprintf(log_file, "<HTML><HEAD>\n<TITLE>%s</TITLE>\n</HEAD><BODY>\n",
	  (*options.logtitle=='\0') ? DFL_LOGTITLE : options.logtitle);
  fflush(log_file);
}

void Log_Cleanup()
{
  if (log_file==NULL)
    return;
  if (log_bquote)
    fputs("</BLOCKQUOTE>", log_file);
  fputs("</BODY></HTML>\n", log_file);
  fclose(log_file);
  log_active = FALSE;
  log_file = NULL;
}

void Log_CommitComment(const char *str) {
  if (!log_active)
    return ;
  fprintf(log_file, "<!-- %s -->\n", str);
  fflush(log_file);
}

void Log_CommitDescription(const char *str) {
  if (!log_active)
    return ;
  if (log_bquote) {
    fputs("</P></BLOCKQUOTE>\n", log_file);
    log_bquote = FALSE;
  }
  fprintf(log_file, "<P>%s</P>\n", str);
  fflush(log_file);
}

void Log_CommitLine(const char *str) {
  if (!log_active)
    return ;
  if (log_bquote)
    fprintf(log_file, "<BR>%s\n", str);
  else {
    fprintf(log_file, "<BLOCKQUOTE>\n<P>%s\n", str);
    log_bquote = TRUE;
  }
  fflush(log_file);
}

void Log_FormatLine(char *out, size_t len, const char *in, TF comment) {
  int c = 0, d = 0;
  TF b = FALSE, i = FALSE, u = FALSE, t = FALSE;
  if (comment)
    while (in[c]!='\0')
      switch(in[c]) {
      case '-':
	if (in[c+1]=='-'&&in[c+2]=='>') {
	  out[d] = out[d+1] = '~';
	  out[d+2] = '>';
	  d+=3;
	  c+=3;
	}
	break;
      case 2:
      case 15:
      case 22:
      case 31:
	c++;
	break;
      default:
	out[d++] = in[c++];
      }
  else {
    while (in[c]!='\0'&&d<len)
      switch(in[c]) {
      case '<':
	if (d+4>=len) {
	  out[d] = '\0';
	  Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	  return ;
	}
	strncpy(out+d, "&lt;", 4);
	d+=4;
	c++;
	break;
      case '>':
	if (d+4>=len) {
	  out[d] = '\0';
	  Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	  return ;
	}
	strncpy(out+d, "&gt;", 4);
	d+=4;
	c++;
	break;
      case '&':
	if (d+5>=len) {
	  out[d] = '\0';
	  Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	  return ;
	}
	strncpy(out+d, "&amp;", 5);
	d+=5;
	c++;
	break;
      case 2:
	if (b) {
	  if (d+4>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "</B>", 4);
	  d+=4;
	  c++;
	  b=FALSE;
	} else {
	  if (d+3>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "<B>", 3);
	  d+=3;
	  c++;
	  b=TRUE;
	}
	break;
      case 22: /* Actual reverse video may not be desirable? Use fixed-width */
	if (t) {
	  if (d+5>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "</TT>", 5);
	  d+=5;
	  c++;
	  t=FALSE;
	} else {
	  if (d+4>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "<TT>", 4);
	  d+=4;
	  c++;
	  t=TRUE;
	}
	break;
      case 31:
	if (u) {
	  if (d+4>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "</U>", 4);
	  d+=4;
	  c++;
	  u=FALSE;
	} else {
	  if (d+3>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "<U>", 3);
	  d+=3;
	  c++;
	  u=TRUE;
	}
	break;
      case 15:
	if (b) {
	  if (d+4>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "</B>", 4);
	  d+=4;
	  b=FALSE;
	}
	if (u) {
	  if (d+4>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "</U>", 4);
	  d+=4;
	  u=FALSE;
	}
	if (t) {
	  if (d+5>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "</TT>", 5);
	  d+=5;
	  t=FALSE;
	}
	c++;
	break;
      case '*':
	if (i) {
	  if (in[c+1]!=' '&&in[c+1]!='.'&&in[c+1]!='!'&&in[c+1]!='?'&&
	      in[c+1]!='\0'&&in[c+1]!='('&&in[c+1]!=')')
	    goto nml;
	  if (d+4>len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "</I>", 4);
	  d+=4;
	  c++;
	  i=FALSE;
	} else {
	  if (c!=0&&in[c-1]!=' '&&in[c-1]!='.'&&in[c-1]!='('&&in[c-1]!=')')
	    goto nml;
	  if (d+3>=len) {
	    out[d] = '\0';
	    Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	    return ;
	  }
	  strncpy(out+d, "<I>", 3);
	  d+=3;
	  c++;
	  i=TRUE;
	}
	break;
      default:
      nml:
	out[d++] = in[c++];
      }
    if (d>=len) {
      out[d] = '\0';
      Util_Eputs("Log: ALLOC_BUFFER too small!\n");
      return ;
    }
    if (b) {
      if (d+4>=len) {
	out[d] = '\0';
	Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	return ;
      }
      strncpy(out+d, "</B>", 4);
      d+=4;
    }
    if (i) {
      if (d+4>=len) {
	out[d] = '\0';
	Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	return ;
      }
      strncpy(out+d, "</I>", 4);
      d+=4;
    }
    if (u) {
      if (d+4>=len) {
	out[d] = '\0';
	Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	return ;
      }
      strncpy(out+d, "</U>", 4);
      d+=4;
    }
    if (t) {
      if (d+5>=len) {
	out[d] = '\0';
	Util_Eputs("Log: ALLOC_BUFFER too small!\n");
	return ;
      }
      strncpy(out+d, "</TT>", 5);
      d+=5;
    }
  }
  out[d] = '\0';
}

int Log_ParseType(char *str)
{
  int slen;
  switch(*str) {
  case '(':
    slen=strlen(str);
    if (str[slen-1]==')') {
      int c;
      str[slen-1] = '\0';
      for(c=1;str[c]!='\0';c++)
	str[c-1] = str[c];
      str[c-1] = '\0';
    } else if (strncasecmp(str+1, "OOC)", 4)==0) {
      int c,d;
      for (c=6;str[c]!='\0';c++)
	str[c-6] = str[c];
      for (d=c-6;d<c;d++)
	str[d] = '\0';
    } else
      return LOG_LINE;
    goto ooc;
  case 'O':
  case 'o':
    if (strncasecmp(str+1, "OC:", 3)==0) {
      int c,d;
      for (c=5;str[c]!='\0';c++)
	str[c-5] = str[c];
      for (d=c-5;d<c;d++)
	str[d] = '\0';
    } else
      return LOG_LINE;
  ooc:
    if (strncasecmp(str, "fix:", 4)==0)
      return LOG_FIX;
    else
      return LOG_OOC;
  case '<':
    slen = strlen(str);
    if (str[slen-1]=='>') {
      int c;
      str[slen-1] = '\0';
      for (c=1;str[c]!='\0';c++)
	str[c-1] = str[c];
      str[c-1] = '\0';
      return LOG_DESC;
    }
  default:
    return LOG_LINE;
  }
}

void Log_Public(Auth_PUser user, const char *nick, const char *str)
{
  int type;
  char line[ALLOC_BUFFER], out[ALLOC_BUFFER];

  Util_Copy(line, str, ALLOC_BUFFER);
  type = Log_ParseType(line);
  switch (type) {
  case LOG_OOC:
    if (options.logooc) {
      Log_FormatLine(out, ALLOC_BUFFER - strlen(nick) - 8, line, TRUE);
      snprintf(line, ALLOC_BUFFER, "OOC - %s: %s", nick, out);
      Log_CommitComment(line);
    }
    break;
  case LOG_FIX:
    Log_FormatLine(out, ALLOC_BUFFER - strlen(nick) - 3, line, TRUE);
    snprintf(line, ALLOC_BUFFER, "%s - %s", nick, out);
    Log_CommitComment(line);
    break;
  case LOG_DESC:
    if (user!=NULL) {
      if (user!=Auth_TUser::GetMaster() && !options.nodesccmt) {
	snprintf(out, ALLOC_BUFFER, "desc from user %s", user->GetAcct());
	Log_CommitComment(out);
      }
    } else {
      snprintf(out, ALLOC_BUFFER, "desc from nick %s", nick);
      Log_CommitComment(out);
    }
    Log_FormatLine(out, ALLOC_BUFFER, line, FALSE);
    Log_CommitDescription(out);
    break;
  case LOG_LINE:
    Log_FormatLine(out, ALLOC_BUFFER - strlen(nick) - 2, line, FALSE);
    snprintf(line, ALLOC_BUFFER, "%s: %s", nick, out);
    Log_CommitLine(line);
  }
}

void Log_Action(const char *nick, const char *str)
{
  int type;
  char line[ALLOC_BUFFER], out[ALLOC_BUFFER];

  Util_Copy(line, str, ALLOC_BUFFER);
  type = Log_ParseType(line);
  switch (type) {
  case LOG_OOC:
    if (options.logooc) {
      Log_FormatLine(out, ALLOC_BUFFER - strlen(nick) - 7, line, TRUE);
      snprintf(line, ALLOC_BUFFER, "OOC - %s %s", nick, out);
      Log_CommitComment(line);
    }
    break;
  case LOG_FIX:  /* People shouldn't do this, but we'll support anyway */
    Log_FormatLine(out, ALLOC_BUFFER - strlen(nick) - 3, line, TRUE);
    snprintf(line, ALLOC_BUFFER, "%s - %s", nick, out);
    Log_CommitComment(line);
    break;
  case LOG_DESC: /* This *really* shouldn't happen */
    Util_Copy(line, str, ALLOC_BUFFER);
  case LOG_LINE:
    Log_FormatLine(out, ALLOC_BUFFER - strlen(nick) - 1, line, FALSE);
    snprintf(line, ALLOC_BUFFER, "%s %s", nick, out);
    Log_CommitLine(line);
  }
}

