/*
 * Lachesis - Interface to filesystem
 * Copyright 2002-2004 M. Dennis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include "auth.h"
#include "lachesis.h"
#include "alloc.h"
#include "types.h"
#include "utils.h"
#include "config.h"

static char* id="@(#) $Id: loaddata.cc,v 1.1.1.1.2.1 2004/03/17 05:35:53 lachesis Exp $";

extern CmdOpts options;

TF LD_AuthGetUsers()
{
  struct dirent *dirlist, *flglist;
  DIR *dir, *flg;
  FILE *in;
  Auth_PUser tuser;
  char *input = NULL, *masteracct = NULL, *path, val;
  char acct[ALLOC_ACCT], pass[ALLOC_PASS];
  size_t len = 0;
  bool tbl;
  path = Util_FormatA("%s/master", options.datadir);
  in = fopen(path, "r");
  free(path);
  if (in!=NULL) {
    getline(&masteracct, &len, in);
    *strchr(masteracct, '\n') = '\0';
    fclose(in);
  } else {
    Util_Eputs("LoadData: No Master account specified! Failed...\n");
    return FALSE;
  }
  len = 0;
  path = Util_FormatA("%s/users", options.datadir);
  dir = opendir(path);
  free(path);
  if (dir==NULL) {
    Util_Eprintf("LoadData: Unexpected error from opendir: %m\n");
    free (masteracct);
    return FALSE;
  }
  while ((dirlist=readdir(dir))!=NULL) {
    if (dirlist->d_type!=DT_REG || !Util_Isalnum(dirlist->d_name))
      continue;
#ifdef RPGSERV
    if (*options.single)
      if (strcasecmp(dirlist->d_name, masteracct) &&
	  strcasecmp(dirlist->d_name, options.single))
	continue;
#endif
    Util_Copy(acct, dirlist->d_name, ALLOC_ACCT);
    path = Util_FormatA("%s/users/%s", options.datadir, acct);
    in = fopen(path, "r");
    free(path);
    if (in==NULL) {
      Util_Eprintf("LoadData: Error retrieving password for %s: %m\n",
		   acct);
      free (masteracct);
      return FALSE;
    }
    if (getline(&input, &len, in)<1) {
      Util_Eprintf("LoadData: Error retrieving password for %s: empty file\n",
		   acct);
      free (masteracct);
      if (input != NULL)
	free (input);
      return FALSE;
    }
    path = strchr(input, '\n');
    if (path!=NULL)
      *path = '\0';
    fclose(in);
    if (!isxdigit(input[0])) {
      Util_Eprintf("LoadData: Error retrieving settings for %s\n", acct);
      free (masteracct);
      free (input);
      return FALSE;
    }
    tbl = strcasecmp(acct, masteracct)==0;
    tuser = new Auth_TUser(acct, input+1, tbl!=FALSE);
    if (tuser!=NULL) {
      val = Util_hex2dec(*input);
      tuser->security = !(val&1);
      tuser->usemsg = !!(val&2);
#ifndef RPGSERV
      tuser->sounds = !(val&4);
      tuser->debug = !!(val&8);
#endif
    }
  }
  free(masteracct);
  free(input);
  if (Auth_TUser::GetMaster()==NULL) {
    Util_Eputs("LoadData: No account is master! Failed...\n");
    return FALSE;
  }
  return TRUE;
}

char LD_ReadSettings(const char *acct)
{
  char *path, hex;
  int fd, err;
  path = Util_FormatA("%s/users/%s", options.datadir, acct);
  fd = open(path, O_RDONLY);
  if (fd<0) {
    Util_Eprintf("LoadData: Unable to open settings file: %m\n");
    free(path);
    return '\0';
  }
  free(path);
  err = pread(fd, &hex, 1, 0);
  if (err==0) {
    Util_Eputs("LoadData: Unknown error reading settings.\n");
    close(fd);
    return '\0';
  } else if (err==-1) {
    Util_Eprintf("LoadData: Error reading settings: %m\n");
    close(fd);
    return '\0';
  }
  close(fd);
  if (isxdigit(hex)) return hex;
  Util_Eputs("LoadData: Garbage data reading settings.\n");
  return '\0';
}

void LD_WriteSettings(const char *acct, char mask)
{
  char *path, val, hex;
  int fd, err;
  if ((hex = LD_ReadSettings(acct))=='\0')
    return ;
  val = Util_hex2dec(hex);
  if (val<0) {
    Util_Eputs("LoadData: Error reading settings.\n");
    return ;
  }
  hex = Util_dec2hex((mask&128)?val&mask:val|mask&15);
  if (!isxdigit(hex)) {
    Util_Eputs("LoadData: Error calculating setting.\n");
    return ;
  }
  path = Util_FormatA("%s/users/%s", options.datadir, acct);
  fd = open(path, O_WRONLY);
  if (fd<0) {
    Util_Eprintf("Error opening settings for writing: %m\n");
    free(path);
    return ;
  }
  free(path);
  do
    err = pwrite(fd, &hex, 1, 0);
  while (err==0 || (err==-1 && errno==EINTR));
  if (err<0)
    Util_Eprintf("LoadData: Error writing setting: %m\n");
  close(fd);
}

void LD_SetSecurity(const char *acct, TF setting)
{
  LD_WriteSettings(acct, setting?~1:1);
}

void LD_SetUseMSG(const char *acct, TF setting)
{
  LD_WriteSettings(acct, setting?2:~2);
}

#ifndef RPGSERV
void LD_SetSounds(const char *acct, TF setting)
{
  LD_WriteSettings(acct, setting?~4:4);
}

#endif
void LD_SetPassword(const char *acct, const char *pass)
{
  // Note: Validity check must occur before this call is made.
  char *path, hex;
  FILE *out;
  int c=0;
  hex = LD_ReadSettings(acct);
  if (hex=='\0') {
    Util_Eputs("LoadData: Aborting password write.\n");
    return ;
  }
  path = Util_FormatA("%s/users/%s", options.datadir, acct);
  do {
    c++;
    if (c>10) {
      Util_Eprintf("File error: unable to write password: %m\n");
      free(path);
      return ;
    }
    unlink(path);
  } while ((out=fopen(path, "wx"))==NULL);
  free(path);
  fprintf(out, "%c%s\n", hex, pass);
  fclose(out);
}

