/*
 * Various utility functions
 */

#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include "alloc.h"
#include "utils.h"
#include "lachesis.h"
#include "config.h"

extern CmdOpts options;

char util_internal_buffer[ALLOC_BUFFER];
const char util_casediff = 'a' - 'A';

#include "utsep.c"

/* g++ 2.95.3 is not liking getline, so we provide a wrapper. */
// TODO: Might this be affected by _GNU_SOURCE? Test.
long Util_Getline(char **LINEPTR, size_t *N, FILE *STREAM)
{
  return getline(LINEPTR, N, STREAM);
}

void Util_Strip(char *buf, unsigned int len, TF flag)
{
  int in = 0, out = 0;
  while((buf[in]>='0'&&buf[in]<='9' || buf[in]>='A'&&buf[in]<='Z' ||
	 buf[in]>='a'&&buf[in]<='z') && in<len) {
    if (flag&&buf[in]>='A'&&buf[in]<='Z')
      buf[out] += util_casediff;
    in++;
    out++;
  }
  if (in==len||buf[in]=='\0')
    return;
  in++;
  while(1) {
    while((buf[in]>='0'&&buf[in]<='9' || buf[in]>='A'&&buf[in]<='Z' ||
	   buf[in]>='a'&&buf[in]<='z') && in<len) {
      if (flag&&buf[in]>='A'&&buf[in]<='Z')
	buf[out++] = buf[in++] + util_casediff;
      else
	buf[out++] = buf[in++];
    }
    if (in==len||buf[in]=='\0') {
      buf[out] = '\0';
      return;
    }
    in++;
  }
}

const char * Util_Separate(const char **data, char delim)
{
  const char *temp;
  int cmdlen, c;
  temp = strchr(*data, delim);
  if (temp==NULL)
    return NULL;
  cmdlen = temp - *data;
  if (cmdlen>=ALLOC_BUFFER)
    cmdlen=ALLOC_BUFFER-1;
  for (c=0;c<cmdlen;c++)
    util_internal_buffer[c] = (*data)[c];
  util_internal_buffer[cmdlen] = 0;
  *data = temp + 1;
  return util_internal_buffer;
}

void Util_Explode(const char *data, char delim, const char **ptr, int len)
{
  char *temp;
  int c;
  if (len<1)
    return ;
  Util_Copy(util_internal_buffer, data, ALLOC_BUFFER);
  temp = util_internal_buffer;
  *ptr = util_internal_buffer;
  for (c=1;c<len;c++) {
    temp = strchr(temp, delim);
    if (!temp) {
      int d;
      for (d=c;d<len;d++)
	ptr[d] = NULL;
      return;
    }
    *temp = '\0';
    temp++;
    ptr[c] = temp;
  }
}

int Util_Count(const char *data, char delim)
{
  int c=0;
  const char *temp = data;

  if (!delim)
    return 1;

  while ((temp = strchr(temp, delim))!=NULL) {
    c++;
    temp++;
  }

  return c;
}

void Util_Cat(char *dst, const char *src, unsigned int size)
{
  unsigned int len;
  char *buf;
  len = size - strlen(dst);
  buf = strchr(dst, 0);
  Util_Copy(buf, src, len);
}

void Util_Eputs(const char *str)
{
  fputs(str, stderr);
  if (options.errlog!=NULL)
    fputs(str, options.errlog);
}

void Util_Eprintf(const char *fmt, ...)
{
  va_list ap;
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
  if (options.errlog!=NULL) {
    va_start(ap, fmt);
    vfprintf(options.errlog, fmt, ap);
    va_end(ap);
  }
}

char * Util_FormatA(const char *fmt, ...)
{
  va_list ap;
  int res;
  char *temp;
  va_start(ap, fmt);
  res = vasprintf(&temp, fmt, ap);
  va_end(ap);
  if (res<0) {
    Util_Eputs("Util_FormatA: Allocation error.\n");
    return NULL;
  }
  return temp;
}

const char * Util_Format(const char *fmt, ...)
{
  va_list ap;
  int res;
  va_start(ap, fmt);
  res = vsnprintf(util_internal_buffer, ALLOC_BUFFER, fmt, ap);
  va_end(ap);
  if (res>=ALLOC_BUFFER) {
    Util_Eputs("Util_Format: Buffer allocation is insufficient; message truncated.\n");
    util_internal_buffer[ALLOC_BUFFER-1] = 0;
  }
  return util_internal_buffer;
}

const char * Util_FormatD(const char *fmt, const char *arg1, const char *arg2,
			  const char *arg3)
{
  // Wrapper for dealing with situations where one of the arguments may be
  // our own buffer.
  // Note: This should probably be deprecated in favor of Util_FormatA...
  char *temp;
  if (arg1==NULL)
    return fmt;
  if (arg1==util_internal_buffer) {
    temp = strdup(arg1);
    Util_Format(fmt, temp, arg2, arg3);
    free (temp);
    return util_internal_buffer;
  }
  if (arg2==util_internal_buffer) {
    temp = strdup(arg2);
    Util_Format(fmt, arg1, temp, arg3);
    free (temp);
    return util_internal_buffer;
  }
  if (arg3==util_internal_buffer) {
    temp = strdup(arg3);
    Util_Format(fmt, arg1, arg2, temp);
    free (temp);
    return util_internal_buffer;
  }
  return Util_Format(fmt, arg1, arg2, arg3);
}

const char * Util_CTCPFmt(const char *arg1, const char *arg2)
{
  char *temp;
  if (arg1==NULL)
    return NULL;
  util_internal_buffer[0] = 1;
  util_internal_buffer[1] = 0;
  Util_Cat(util_internal_buffer, arg1, ALLOC_BUFFER);
  if (arg2!=NULL) {
    temp = strchr(util_internal_buffer, 0);
    if (temp - util_internal_buffer >= ALLOC_BUFFER - 2) {
      Util_Eputs("Util_CTCPFmt: Buffer allocation is insufficient; message truncated.\n");
      util_internal_buffer[ALLOC_BUFFER-2] = 1;
      return util_internal_buffer;
    }
    temp[0] = 32;
    temp[1] = 0;
    Util_Cat(util_internal_buffer, arg2, ALLOC_BUFFER);
  }
  temp = strchr(util_internal_buffer, 0);
  if (util_internal_buffer - temp >= ALLOC_BUFFER - 1) {
    Util_Eputs("Util_CTCPFmt: Buffer allocation is insufficient; message truncated.\n");
    util_internal_buffer[ALLOC_BUFFER-2] = 1;
    return util_internal_buffer;
  }
  temp[0] = 1;
  temp[1] = 0;
  return util_internal_buffer;
}

const char * Util_CTCPFmtD(const char *arg1, const char *arg2)
{
  char *temp;
  if (arg1==NULL)
    return NULL;
  if (arg1==util_internal_buffer) {
    temp = strdup(arg1);
    Util_CTCPFmt(temp, arg2);
    free (temp);
    return util_internal_buffer;
  }
  if (arg2==util_internal_buffer) {
    temp = strdup(arg2);
    Util_CTCPFmt(arg1, temp);
    free (temp);
    return util_internal_buffer;
  }
  return Util_CTCPFmt(arg1, arg2);
}
