/*
 * Lachesis, an IRCRPG combat engine - Native IRC Interface
 * Copyright 2002-2003 M. Dennis
 *
 * This interface creates a native low-level interface to an IRC server.
 * It also handles DCC CHAT connections. At this time,  DCC SEND is not
 * supported.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "irc-numeric.h"
#include "fdio.h"
#include "lachesis.h"
#include "types.h"
#include "irc.h"
#include "utils.h"
#include "alloc.h"
#include "ircint.h"
#include "config.h"

fd_set *lin_read_fds = NULL;
int lin_fd = -1;
char lin_our_nick[ALLOC_NICK];
char lin_our_name[ALLOC_NAME];
char lin_our_user[ALLOC_USER];
char lin_our_host[ALLOC_HOST];
char lin_prefix[ALLOC_MINIBUFFER];
char lin_our_server[ALLOC_HOST];
char *lin_input = NULL;
size_t lin_input_len = 0;
TF lin_active = FALSE;
TF lin_registering = FALSE;
unsigned long lin_our_addr;
extern CmdOpts options;

typedef struct tag_dcc_ticket {
  char nick[ALLOC_NICK];
  int sock;
  TF listen;
  TF stale;
  struct sockaddr_in addr;  // To whom we are connected (or will connect)
  socklen_t addrlen;
  struct tag_dcc_ticket *next, *prev;
} TDCC, *PDCC;

PDCC dcc_list = NULL;

void AutoChangeNick()
{
  char *ptr;
  ptr = strchr(lin_our_nick, '\0');
  if (ptr-lin_our_nick<ALLOC_NICK-1) {
    ptr[0] = '_';
    ptr[1] = '\0';
  } else {
    char temp = *--ptr;
    while (ptr!=lin_our_nick) {
      --ptr;
      ptr[1] = ptr[0];
    }
    ptr[1] = ptr[0];
    ptr[0] = temp;
  }
  fdprintf(lin_fd, "NICK %s\r\n", lin_our_nick);
}

void RemDCC(PDCC ticket)
{
  if (ticket == NULL)
    return ;
  if (ticket->sock != -1) {
    FD_CLR(ticket->sock, lin_read_fds);
    close (ticket->sock);
  }
  if (dcc_list == ticket)
    dcc_list = ticket->next;
  if (ticket->next!=NULL)
    ticket->next->prev = ticket->prev;
  if (ticket->prev!=NULL)
    ticket->prev->next = ticket->next;
  free (ticket);
}

void NewDCC(const char *nick, int fd)
{
  PDCC temp;
  temp = (PDCC)malloc(sizeof(TDCC));
  temp->addrlen = sizeof (struct sockaddr_in);
  temp->prev = NULL;
  temp->next = dcc_list;
  if (dcc_list!=NULL)
    dcc_list->prev = temp;
  Util_Copy(temp->nick, nick, ALLOC_NICK);
  dcc_list = temp;
  temp->sock = fd;
  if (fd>-1) {
    FD_SET(fd, lin_read_fds);
    temp->listen = TRUE;
  } else
    temp->listen = FALSE;
  temp->stale = FALSE;
}

void DoDCCCleanup() {
  PDCC t1, t2 = dcc_list;
  while (t2 != NULL) {
    t1 = t2->next;
    if (t2->stale)
      RemDCC(t2);
    t2 = t1;
  }
}

void AcceptDCC(PDCC dcc)
{
  int asock, ioflags;

  if (dcc->sock<0 || !dcc->listen) return ;
  asock = accept(dcc->sock, (struct sockaddr *)&(dcc->addr), 
		 &(dcc->addrlen));
  if (asock<0) {
    Util_Eprintf("AcceptDCC: Unable to accept connection: %m\n");
    return ;
  }

  //  ioflags=fcntl(asock, F_GETFL, 0);
  //  ioflags|=O_NONBLOCK;
  //  fcntl(asock, F_SETFL, ioflags);

  FD_CLR(dcc->sock, lin_read_fds);
  FD_SET(asock, lin_read_fds);
  close(dcc->sock);
  dcc->sock = asock;
  dcc->listen = FALSE;
  HandleDCCChatOpened((void *)dcc);
}

#if 0
PDCC GetDCC(const char *nick) // Is this even needed?
{
  PDCC temp = dcc_list;
  while (temp!=NULL) {
    if (strcmp(temp->nick, nick)==0)
      return temp;
    temp = temp->next;
  }
  return NULL;
}
#endif

void PrefixSeparate()
{
  char *temp;
  if ((temp=strchr(lin_prefix, '@')) != NULL)
    *temp = 0;
  if ((temp=strchr(lin_prefix, '!')) != NULL)
    *temp = 0;
}

void IRC_Userhost(const char *target)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC USERHOST %s\n", target);
#endif
  fdprintf(lin_fd, "USERHOST %s\r\n", target);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Quote(const char *str, TF priority)
{
  // priority is currently unused.
#ifdef DEBUG
  Util_Eprintf("DEBUG - Sending to server: %s\n", str);
#endif
  fdprintf(lin_fd, "%s\r\n", str);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Notice(const char *target, const char *message)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC NOTICE %s %s\n", target, message);
#endif
  fdprintf(lin_fd, "NOTICE %s :%s\r\n", target, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Part(const char *channel, const char *message)
{
  if (message!=NULL&&*message) {
#ifdef DEBUG
    Util_Eprintf("DEBUG - IRC PART %s :%s\n", channel, message);
#endif
    fdprintf(lin_fd, "PART %s :%s\r\n", channel, message);
  } else {
#ifdef DEBUG
    Util_Eprintf("DEBUG - IRC PART %s\n", channel);
#endif
    fdprintf(lin_fd, "PART %s\r\n", channel);
  }
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_CTCP(const char *target, const char *command, const char *data)
{
  if (data!=NULL) {
#ifdef DEBUG
    Util_Eprintf("DEBUG - IRC CTCP %s %s %s\n", target, command, data);
#endif
    fdprintf(lin_fd, "PRIVMSG %s :%c%s %s%c\r\n", target, 1, command, data, 1);
  } else {
#ifdef DEBUG
    Util_Eprintf("DEBUG - IRC CTCP %s %s\n", target, command);
#endif
    fdprintf(lin_fd, "PRIVMSG %s :%c%s%c\r\n", target, 1, command, 1);
  }
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_CTCPReply(const char *target, const char *command, const char *data)
{
  IRC_Notice(target, Util_CTCPFmtD(command, data));
}

const char * IRC_VersionInfo()
{
  return "Using native IRC interface.";
}

void IRC_Nick(const char *nick)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC NICK %s\n", nick);
#endif
  fdprintf(lin_fd, "NICK %s\r\n", nick);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Public(const char *channel, const char *message)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC PUBLIC %s %s\n", channel, message);
#endif
  fdprintf(lin_fd, "PRIVMSG %s :%s\r\n", channel, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Action(const char *channel, const char *message)
{
  IRC_CTCP(channel, "ACTION", message);
}

void IRC_MSG(const char *target, const char *message)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC MSG %s %s\n", target, message);
#endif
  fdprintf(lin_fd, "PRIVMSG %s :%s\r\n", target, message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Describe(const char *target, const char *message)
{
  IRC_CTCP(target, "ACTION", message);
}

void IRC_Wallop(const char *message)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC WALLOP %s\n", message);
#endif
  fdprintf(lin_fd, "WALLOPS :%s\r\n", message);
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Join(const char *channel, const char *key)
{
  if (key!=NULL) {
#ifdef DEBUG
    Util_Eprintf("DEBUG - IRC JOIN %s %s\n", channel, key);
#endif
    fdprintf(lin_fd, "JOIN %s %s\r\n", channel, key);
  } else {
#ifdef DEBUG
    Util_Eprintf("DEBUG - IRC JOIN %s\n", channel);
#endif
    fdprintf(lin_fd, "JOIN %s\r\n", channel);
  }
  //  FD_SET(fileno(icsout), write_fds);
}

void IRC_Quit(const char *message)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC QUIT %s\n", message);
#endif
  fdprintf(lin_fd, "QUIT :%s\r\n", message);
  IRC_Shutdown();
}

void IRC_Shutdown() {
#ifdef DEBUG
  Util_Eputs("DEBUG - Closing IRC server connection\n");
#endif
  close(lin_fd);
  lin_fd = -1;
  Shutdown();
}

TF IRC_IsConnected()
{
  return lin_active;
}

const char * IRC_GetNick()
{
  return lin_our_nick;
}

const char * IRC_GetHost()
{
  return lin_our_host;
}

const char * IRC_HostmaskNick(const char *prefix)
{
  if (prefix!=lin_prefix)
    strncpy(lin_prefix, prefix, ALLOC_MINIBUFFER-1);
  if (strchr(lin_prefix, '!')!=NULL)
    PrefixSeparate();
  return lin_prefix;
}

const char * IRC_HostmaskUser(const char *prefix)
{
  const char *temp;
  if (prefix!=lin_prefix)
    strncpy(lin_prefix, prefix, ALLOC_MINIBUFFER-1);
  if (strchr(lin_prefix, '!')!=NULL)
    PrefixSeparate();
  temp = strchr(lin_prefix, 0);
  return temp+1;
}

const char * IRC_HostmaskHost(const char *prefix)
{
  const char *temp;
  if (prefix!=lin_prefix)
    strncpy(lin_prefix, prefix, ALLOC_MINIBUFFER-1);
  if (strchr(lin_prefix, '!')!=NULL)
    PrefixSeparate();
  temp = strchr(lin_prefix, 0);
  temp = strchr(temp+1, 0);
  return temp+1;
}

void * IRC_DCCChatSend(const char *target)
{
  int sock;
  struct sockaddr_in sin;
  size_t len;
  char buffer[ALLOC_MINIBUFFER];

#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC DCC CHAT %s\n", target);
#endif
  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock<0) {
    Util_Eprintf("IRC_DCCChatSend: Unable to create socket: %m\n");
    return NULL;
  }
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = htonl(INADDR_ANY);
  sin.sin_port = 0;
  len = 1;
  setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (void *)(&len), sizeof(len));
  if (bind(sock, (struct sockaddr *)(&sin), sizeof(struct sockaddr_in))<0) {
    Util_Eprintf("IRC_DCCChatSend: Unable to bind socket: %m\n");
    close (sock);
    return NULL;
  }
  len = sizeof(struct sockaddr_in);
  if (getsockname(sock, (struct sockaddr *)(&sin), &len)<0) {
    Util_Eprintf("IRC_DCCChatSend: Unable to read address: %m\n");
    close (sock);
    return NULL;
  }
  if (listen(sock, 1)<0) {
    Util_Eprintf("IRC_DCCChatSend: Unable to listen: %m\n");
    close (sock);
    return NULL;
  }
  NewDCC(target, sock);
  snprintf(buffer, ALLOC_MINIBUFFER, "CHAT chat %lu %d", 
	   ntohl(lin_our_addr), ntohs(sin.sin_port));
  IRC_CTCP(target, "DCC", buffer);
  return (void *)dcc_list;
}

void IRC_DCCChatAccept(void *cookie)
{
  int sock, ioflags;

#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC DCC CHAT ACCEPT %s\n", ((PDCC)cookie)->nick);
#endif
  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock<0) {
    Util_Eprintf("IRC_DCCChatAccept: Unable to create socket: %m\n");
    return ;
  }
  if (connect(sock, (struct sockaddr *)&(((PDCC)cookie)->addr), 
              ((PDCC)cookie)->addrlen)<0) {
    Util_Eprintf("IRC_DCCChatAccept: Unable to connect: %m\n");
    close (sock);
    return ;
  }

  //  ioflags=fcntl(sock, F_GETFL, 0);
  //  ioflags|=O_NONBLOCK;
  //  fcntl(sock, F_SETFL, ioflags);

  ((PDCC)cookie)->sock = sock;
  FD_SET(sock, lin_read_fds);
  HandleDCCChatOpened(cookie);
}

void IRC_DCCChatReject(void *cookie)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC DCC CHAT REJECT %s\n", ((PDCC)cookie)->nick);
#endif
  IRC_CTCPReply(((PDCC)cookie)->nick, "ERRMSG", "DCC CHAT chat declined");
  RemDCC((PDCC)cookie);
}

void IRC_DCCChatMessage(void *cookie, const char *message)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC DCC MSG %s %s\n", ((PDCC)cookie)->nick, message);
#endif
  fdprintf(((PDCC)cookie)->sock, "%s\n", message);
  //  FD_SET(fileno(((PDCC)cookie)->dccout), write_fds);
}

void IRC_DCCChatClose(void *cookie)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC DCC CHAT CLOSE%s\n", ((PDCC)cookie)->nick);
#endif
  RemDCC((PDCC)cookie);
}

const char * IRC_DCCGetNick(void *cookie)
{
  return ((PDCC)cookie)->nick;
}

void IRC_DCCSend(const char *target, const char *fn)
{
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC DCC SEND %s %s\n", target, fn);
#endif
  Util_Eputs("DCC SEND not implemented.\n");
}

void OnDCCChatClose(PDCC dcc)
{
  HandleDCCChatClosed((void *)dcc);
  dcc->stale = TRUE;
}

void OnDCCChatMSG(PDCC dcc, const char *str)
{
  char msg[ALLOC_BUFFER], *temp;
  Util_Copy(msg, str, ALLOC_BUFFER);
  temp = strchr(msg, '\n');
  if (temp != NULL)
    *temp = 0;
  HandleDCCChatMessage((void*)(dcc), msg);
}

#if 0
void OnConnect(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  bn2_our_addr = strtoul(s1, NULL, 10);
  Util_Copy(bn2_our_host, s2, ALLOC_HOST);
  fdprintf(icsfd, "REGISTER %s %s %s\n", bn2_our_nick, bn2_our_user,
	   bn2_our_name);
  //  FD_SET(fileno(icsout), write_fds);
}
#endif

void OnNumeric(const char *str)
{
  char *temp, target[ALLOC_NICK];
  long int num;
  num = strtol(str, &temp, 10);
  if (*temp==' ')
    temp++;
  Util_SepTo(target, ALLOC_NICK, (const char **)&temp, 32);
  switch (num) {
  case RPL_WELCOME:
    if (!lin_registering)
      Util_Eprintf("Received welcome, but we're not registering!\n(%d) %s\n", 
		   num, temp);
    else {
      Util_Copy(lin_our_server, lin_prefix, ALLOC_HOST);
      fdprintf(lin_fd, "USERHOST %s\r\n", lin_our_nick);
    }
    return ;
  case RPL_USERHOST:
    if (lin_registering) {
      char *ptr1, *ptr2;
      char nick[ALLOC_NICK];
      if (*temp==':')
	temp++;
      ptr1 = temp;
      Util_SepTo(nick, ALLOC_NICK, (const char **)&ptr1, '=');
      ptr2 = strchr(nick, '*');
      if (ptr2!=NULL)
	*ptr2 = '\0';
#ifdef DEBUG
      Util_Eprintf("DEBUG - RPL_USERHOST for %s while registering.\n",
		   nick);
#endif
      if (strcasecmp(nick, lin_our_nick)==0) {
	ptr1++;
	Util_SepTo(lin_our_user, ALLOC_USER, (const char **)&ptr1, '@');
	Util_SepTo(lin_our_host, ALLOC_HOST, (const char **)&ptr1, ' ');
	lin_registering=FALSE;
	ConnectionReady();
	return ;
      }
    } else {
      HandleNumeric(num, temp);
      return ;
    }
  case RPL_NAMREPLY: {
    const char *s1, *s2;
    char chan[ALLOC_CHAN];
    s2 = (const char *)temp;
    Util_SepTo(chan, ALLOC_CHAN, &s2, 32);
    if (s2==NULL) {
      Util_Eprintf("Error: Invalid RPL_NAMREPLY message: %s\n", temp);
      return ;
    }
    /* Tiamat seems to pass a character before the channel indicating
       whether it is public, private, or secret. This is against the RFC.
       We compensate by ignoring the first parameter if it does not appear
       to be a chanel. */
    if (*chan!='#' && *chan!='&' && *chan!='+')
      Util_SepTo(chan, ALLOC_CHAN, &s2, 32);
    if (s2==NULL) {
      Util_Eprintf("Error: Invalid RPL_NAMREPLY message: %s\n", temp);
      return ;
    }
    if (*s2 != ':') {
      s1 = Util_Separate(&s2, 32);
      if (s1==NULL)
	s1 = s2;
      HandleName(chan, s1);
      return;
    }
    s2++;
    while ((s1 = Util_Separate(&s2, 32))!=NULL)
      HandleName(chan, s1);
    if (*s2)
      HandleName(chan, s2);
    return ;
  }
  case ERR_ERRONEOUSNICKNAME:
    if (lin_registering) {
      Util_Strip(lin_our_nick, ALLOC_NICK, FALSE);
      fdprintf(lin_fd, "NICK %s\r\n", lin_our_nick);
      return ;
    }
  case ERR_NICKNAMEINUSE:
    if (lin_registering) {
      AutoChangeNick();
      fdprintf(lin_fd, "NICK %s\r\n", lin_our_nick);
      return ;
    }
  case ERR_YOUREBANNEDCREEP:
    Util_Eprintf("Error: Banned from server: (%d) %s\n", num, temp);
    close(lin_fd);
    lin_fd=-1;
    lin_active=FALSE;
    Shutdown();
  case ERR_UNKNOWNCOMMAND:
  case ERR_NOTREGISTERED:
  case ERR_ALREADYREGISTERED:
  case ERR_NOPERMFORHOST:
  case ERR_PASSWDMISMATCH:
  case ERR_NUMERICERROR:
#ifdef DEBUG
    Util_Eprintf("DEBUG: (%d) %s\n", num, temp);
#endif
  default:
    HandleNumeric(num, temp);
  }
}

void OnUnknown(const char *cmd, const char *str)
{
  Util_Eprintf("Unknown event from %s : %s %s\n", lin_prefix, cmd, str);
}

void OnNotice(const char *str)
{
  char target[ALLOC_NICK], *data;
  const char *t1, *t2 = str;
  Util_SepTo(target, ALLOC_NICK, &t2, 32);
  if (*t2 == ':') {
    t2++;
    if (*t2 == 1) {
      char buf[ALLOC_BUFFER], *ptr;
      Util_Copy(buf, t2+1, ALLOC_BUFFER);
      ptr = strchr(buf, 0) - 1;
      if (*ptr == 1)
	*ptr = 0;
      HandleCTCPReply(lin_prefix, buf);
    } else
      HandleNotice(lin_prefix, target, t2);
    return ;
  }
#ifdef DEBUG
  Util_Eprintf("Warning: Notice without final parm %s %s\n", target, t2);
#endif
  t1 = strchr(t2, 32);
  if (t1==NULL) {
    HandleNotice(lin_prefix, target, t2);
    return ;
  }
  data = (char *)malloc(t1 - t2 + 1);
  memcpy(data, t2, t2 - t1);
  data[t2 - t1] = 0;
  HandleNotice(lin_prefix, target, data);
  free(data);
}

void OnDCC(const char *str) {
  if (strncasecmp(str, "CHAT chat ", 10)==0 ||
      strncasecmp(str, "CHAT clear ", 11)==0) {
    unsigned long addr;
    unsigned short port;
    const char *temp;
    PrefixSeparate();
    temp = strchr(str, 32);
    temp = strchr(temp+1, 32);
    addr = strtoul(temp+1, NULL, 10);
    temp = strchr(temp+1, 32);
    port = strtoul(temp+1, NULL, 10);
    if (!addr||!port) {
      char pr[6];
      Util_Copy(pr, strchr(str, 32)+1, 6);
      if (pr[4]==32)
	pr[4] = 0;
      IRC_CTCPReply(lin_prefix, "ERRMSG", 
		    Util_Format("DCC CHAT %s missing or incorrect address", 
				pr, NULL, NULL));
      return ;
    }
    NewDCC(lin_prefix, -1);
    dcc_list->addr.sin_family = AF_INET;
    dcc_list->addr.sin_addr.s_addr = htonl(addr);
    dcc_list->addr.sin_port = htons(port);
#ifdef DEBUG
    Util_Eprintf("DCC CHAT to %s:%u\n", inet_ntoa(dcc_list->addr.sin_addr),
		 port);
#endif
    *strchr(lin_prefix, 0) = '!';
    *strchr(lin_prefix, 0) = '@';
    HandleDCCChatRequestPfx((void *)dcc_list, lin_prefix);
  }
}

void OnCTCP(const char *target, const char *str)
{
  const char *temp;
  if (strncasecmp(str, "DCC ", 4)==0 && strcasecmp(target, lin_our_nick)==0) {
    OnDCC(str+4);
    return ;
  }
  if ((temp = HandleCTCP(lin_prefix, target, str))!=NULL) {
    const char *s1, *s2;
    s2 = str;
    s1 = Util_Separate(&s2, 32);
    if (s1==NULL)
      s1 = s2;
    PrefixSeparate();
    IRC_CTCPReply(lin_prefix, s1, temp);
  }
}

void OnInvite(const char *str)
{
  const char *temp;
  temp = strchr(str, 32) + 1;
  HandleInvite(lin_prefix, temp);
}

void OnKick(const char *str)
{
  char chan[ALLOC_CHAN], nick[ALLOC_NICK];
  const char *temp = str;
  Util_SepTo(chan, ALLOC_CHAN, &temp, 32);
  Util_SepTo(nick, ALLOC_NICK, &temp, 32);
  PrefixSeparate();
  if (temp==NULL) {
    HandleKick(lin_prefix, nick, chan, lin_prefix);
    return ;
  }
  if (*temp==':')
    temp++;
  HandleKick(lin_prefix, nick, chan, temp);
}

void OnMSG(const char *str)
{
  const char *temp = str;
  char target[ALLOC_NICK];
  Util_SepTo(target, ALLOC_NICK, &temp, 32);
  if (temp==NULL)
    return ;
  if (*temp == ':')
    temp++;
#ifdef DEBUG
  else
    Util_Eprintf("Warning: MSG without final parm %s %s\n", lin_prefix, 
		 str);
#endif
  if (*temp == 1) {
    char buf[ALLOC_BUFFER], *ptr;
    Util_Copy(buf, temp+1, ALLOC_BUFFER);
    ptr = strchr(buf, 0) - 1;
    if (*ptr == 1)
      *ptr = 0;
    OnCTCP(target, buf);
  } else {
    if (strcasecmp(target, lin_our_nick)==0)
      HandleMSG(lin_prefix, temp);
    else
      HandlePublic(lin_prefix, target, temp);
  }
}

void OnJoin(const char *str)
{
  int c = strlen(lin_our_nick);
  const char *ptr = str;
  if (*ptr==':')
    ptr++;
  if (lin_prefix[c]=='!' &&
      strncasecmp(lin_prefix, lin_our_nick, c)==0)
    HandleJoined(ptr);
  else
    HandleJoin(lin_prefix, ptr);
}

void OnNick(const char *str)
{
  int c = strlen(lin_our_nick);
  char nick[ALLOC_NICK];
  const char *temp = str;
  if (*temp==':')
    temp++;
  Util_SepTo(nick, ALLOC_NICK, &temp, 32);
  if (lin_prefix[c]=='!' &&
      strncasecmp(lin_prefix, lin_our_nick, c)==0) {
    HandleOurNickChanged(nick);
    Util_Copy(lin_our_nick, nick, ALLOC_NICK);
  } else
    HandleNickChange(lin_prefix, nick);
}

void OnPart(const char *str)
{
  char chan[ALLOC_CHAN];
  const char *temp;
  temp = str;
  Util_SepTo(chan, ALLOC_CHAN, &temp, 32);
  PrefixSeparate();
  if (temp==NULL||*temp==0) {
    HandlePart(lin_prefix, chan, NULL);
    return ;
  }
  if (*temp==':')
    temp++;
  HandlePart(lin_prefix, chan, temp);
}

void OnQuit(const char *str)
{
  const char *temp = str;
  PrefixSeparate();
  if (strcasecmp(lin_prefix, lin_our_nick)==0)
    return ;
  if (*temp==':')
    temp++;
  HandleQuit(lin_prefix, temp);
}

void OnDisconnect(const char *message)
{
  close(lin_fd);
  lin_fd = -1;
  lin_active = FALSE;
  while (dcc_list != NULL)
    RemDCC(dcc_list);
  FD_ZERO(lin_read_fds);
  //  FD_ZERO(write_fds);
  HandleDisconnect(message);
}

void OnError(const char *str)
{
  const char *temp = str;
  if (*temp==':')
    temp++;
  if (strncmp(temp, "Closing Link", 12)==0) {
    char buf[ALLOC_BUFFER];
    temp = strchr(str, '(') + 1;
    Util_Copy(buf, temp, ALLOC_BUFFER);
    *(strchr(buf, ')')) = 0;
    OnDisconnect(buf);
    return;
  }
#ifdef DEBUG
  Util_Eprintf("DEBUG - Received Error: %s\n", temp);
#endif
}

void OnPing(const char *str)
{
  char server[ALLOC_HOST];
  const char *ptr = str;
  if (*ptr==':')
    ptr++;
  Util_SepTo(server, ALLOC_HOST, &ptr, 32);
  if (strcasecmp(server, lin_our_server)==0)
    fdprintf(lin_fd, "PONG %s\r\n", lin_our_nick);
  else
    fdprintf(lin_fd, "PONG %s %s\r\n", lin_our_nick, server);
}

struct strtofunc_map {
  const char *str;
  void (*fun)(const char *);
};

const struct strtofunc_map EVENT_LIST[] = {
  { "ERROR", &OnError },
  { "NOTICE", &OnNotice },
  { "INVITE", &OnInvite },
  { "KICK", &OnKick },
  { "PRIVMSG", &OnMSG },
  { "JOIN", &OnJoin },
  { "NICK", &OnNick },
  { "PART", &OnPart },
  { "QUIT", &OnQuit },
  { "PING", &OnPing },
  { NULL, NULL }
};

void StartMessage()
{
  fd_set reads;
  const char *cmd, *data;
  int c;
  PDCC dcc;
  while (1) {
#ifdef DEBUG
    Util_Eputs("DEBUG - Top of StartMessage loop.\n");
#endif
    memcpy(&reads, lin_read_fds, sizeof(fd_set));
    //    memcpy(&writes, write_fds, sizeof(fd_set));
    select(FD_SETSIZE, &reads, NULL, NULL, NULL);

    for (dcc = dcc_list; dcc != NULL; dcc = dcc->next)
      if (dcc->sock != -1 && FD_ISSET(dcc->sock, &reads)) {
	if (dcc->listen)
	  AcceptDCC(dcc);
	else {
	  int r = fdgetline(&lin_input, &lin_input_len, dcc->sock);
	  if (r<1)
	    OnDCCChatClose(dcc);
	  else {
	    *strchr(lin_input, '\n')=0;
	    OnDCCChatMSG(dcc, lin_input);
	  }
	}
      }
    DoDCCCleanup();

    if (FD_ISSET(lin_fd, &reads)) {
      char *ptr;
      if (fdgetline(&lin_input, &lin_input_len, lin_fd)<0) {
	Util_Eprintf("Error reading socket: %m\n");
	// This is normal for a lost connection. Normal shutdown.
	IRC_Shutdown();
      }
      *(strchr(lin_input, '\n'))=0;
      ptr = strchr(lin_input, '\r');
      if (ptr!=NULL)
	*ptr = 0;
#ifdef DEBUG
      Util_Eprintf("DEBUG - Received from server: %s\n", lin_input);
#endif
      if (lin_input[0]==':') {
	ptr = lin_input + 1;
	Util_SepTo(lin_prefix, ALLOC_MINIBUFFER, (const char **)&ptr, 32);
      } else {
	*lin_prefix=0;
	ptr = lin_input;
      }
      if (ptr[0]>='0'&&ptr[0]<='9')
	OnNumeric(ptr);
      else {
	data = (const char *)(ptr);
	cmd = Util_Separate(&data, 32);
	if (cmd==NULL) {
	  cmd = data;
	  data = NULL;
	}
	for (c=0;1;c++) {
	  if (EVENT_LIST[c].str==NULL) {
	    OnUnknown(cmd, data);
	    break;
	  }
	  if (strcasecmp(cmd, EVENT_LIST[c].str)==0) {
	    (*(EVENT_LIST[c].fun))(data);
	    break;
	  }
	}
      }
    }
  }
}

void IRC_Connect(const char *server, const int port, const char *nick,
		 const char *user, const char *s_name)
{
  int sock, res, ioflags;
  const char *fn;
  struct sockaddr_in sin;
  struct hostent *host;
  socklen_t slen;

#ifdef DEBUG
  Util_Eputs("DEBUG - IRC_Connect called\n");
#endif

  lin_read_fds = (fd_set *)malloc(sizeof(fd_set));
  //  write_fds = (fd_set *)malloc(sizeof(fd_set));
  FD_ZERO(lin_read_fds);
  //  FD_ZERO(write_fds);

  Util_Copy(lin_our_nick, nick, ALLOC_NICK);
  Util_Copy(lin_our_user, user, ALLOC_USER);
  Util_Copy(lin_our_name, s_name, ALLOC_NAME);

  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock<0) {
    Util_Eprintf("Failed to create socket: %m\n");
    exit (1);
  }
  sin.sin_family=AF_INET;
  if (!inet_aton(server, &(sin.sin_addr))) {
    host = gethostbyname2(server, AF_INET);
    if (host==NULL) {
      Util_Eprintf("Unable to resolve host: %s\n", server);
      exit (1);
    }
    sin.sin_addr.s_addr = *((unsigned long *)host->h_addr);
#ifdef DEBUG
    Util_Eprintf("DEBUG - Using resolved address %s\n",
		 inet_ntoa(sin.sin_addr));
#endif
  }
  sin.sin_port = htons(port);
  if (connect(sock, (struct sockaddr *)&sin, sizeof(sin))<0) {
    Util_Eprintf("Failed to connect: %m\n");
    exit (1);
  }
  slen = sizeof(sin);
  getsockname(sock, (struct sockaddr *)&sin, &slen);
  lin_our_addr = sin.sin_addr.s_addr;

  //  ioflags=fcntl(sock, F_GETFL, 0);
  //  ioflags|=O_NONBLOCK;
  //  fcntl(sock, F_SETFL, ioflags);

  FD_SET(sock, lin_read_fds);
  lin_fd = sock;

  lin_registering = TRUE;
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC NICK %s\n", lin_our_nick);
#endif
  fdprintf(lin_fd, "NICK %s\r\n", lin_our_nick);
  host = gethostbyaddr((const char *)&lin_our_addr, 4, AF_INET);
#ifdef DEBUG
  Util_Eprintf("DEBUG - IRC USER %s %s %s :%s\n", lin_our_user,
	       host->h_name, server, lin_our_name);
#endif
  fdprintf(lin_fd, "USER %s %s %s :%s\r\n", lin_our_user, host->h_name,
	   server, lin_our_name);

  StartMessage();
}

void LowIrcCleanup()
{
#ifdef DEBUG
  Util_Eputs("DEBUG - LowIrcCleanup called\n");
#endif
  if (lin_input!=NULL) {
    free(lin_input);
    lin_input=NULL;
    lin_input_len=0;
  }
  if (lin_read_fds!=NULL) {
    free (lin_read_fds);
    lin_read_fds = NULL;
  }
  //  if (write_fds!=NULL) {
  //    free (write_fds);
  //    write_fds = NULL;
  //  }
  if (lin_fd!=-1) {
    close(lin_fd);
    lin_fd = -1;
  }
  while (dcc_list != NULL)
    RemDCC(dcc_list);
}
