/*
 * Lachesis, an IRCRPG combat engine - Message-based IRC Interface
 * Copyright 2002 M. Dennis
 *
 * This interface will provide Lachesis as a service to an existing
 * IRC bot. This is a quick hack until a direct interface is more
 * viable.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include "lachesis.h"
#include "types.h"
#include "irc.h"
#include "utils.h"
#include "alloc.h"
#include "ircint.h"

FILE *our_conn;
char our_nick[ALLOC_NICK];
char our_name[ALLOC_NAME];
char our_user[ALLOC_USER];
char our_host[ALLOC_HOST];
char internal_buffer[ALLOC_MINIBUFFER];

extern CmdOpts options;

typedef struct tag_dcc_ticket {
  char nick[ALLOC_NICK];
  struct tag_dcc_ticket *next, *prev;
} TDCC, *PDCC;

PDCC dcc_list = NULL;

void RemDCC(PDCC ticket)
{
  if (ticket == NULL)
    return ;
  if (dcc_list == ticket)
    dcc_list = ticket->next;
  if (ticket->next!=NULL)
    ticket->next->prev = ticket->prev;
  if (ticket->prev!=NULL)
    ticket->prev->next = ticket->next;
}

void NewDCC(const char *nick)
{
  PDCC temp;
  temp = (PDCC)malloc(sizeof(TDCC));
  temp->prev = NULL;
  temp->next = dcc_list;
  if (dcc_list!=NULL)
    dcc_list->prev = temp;
  Util_Copy(temp->nick, nick, ALLOC_NICK);
  dcc_list = temp;
}

PDCC GetDCC(const char *nick)
{
  PDCC temp = dcc_list;
  while (temp!=NULL) {
    if (strcasecmp(temp->nick, nick)==0)
      return temp;
    temp = temp->next;
  }
  return NULL;
}

void PrefixSeparate()
{
  *(strchr(internal_buffer, '!')) = 0;
  *(strchr(internal_buffer, '@')) = 0;
}

void PrefixGenerate(const char **data)
{
  const char *temp;
  temp = strchr(*data, 32);
  temp = strchr(++temp, 32);
  temp++;
  if (temp - *data > ALLOC_MINIBUFFER) {
    fputs("MsgIrc: Prefix overflow - Increase size of ALLOC_MINIBUFFER", stderr);
    fprintf(stderr, "Buffer: %s", *data);
    exit (255);
  }
  strncpy(internal_buffer, *data, temp - *data);
  *(strchr(internal_buffer, 32)) = '!';
  *(strchr(internal_buffer, 32)) = 0;
  *data = temp;
}

void IRC_Userhost(const char *target)
{
  fprintf(our_conn, "USERHOST %s\n", target);
}

void IRC_Quote(const char *command, const char *data, TF priority)
{
  if (data!=NULL)
    fprintf(our_conn, "DO QUOTE %s :%s\n", command, data);
  else
    fprintf(our_conn, "DO QUOTE %s\n", command);
}

void IRC_Notice(const char *target, const char *message)
{
  fprintf(our_conn, "DO NOTICE %s %s\n", target, message);
}

void IRC_Part(const char *channel, const char *message)
{
  if (message!=NULL&&*message)
    fprintf(our_conn, "DO QUOTE PART %s :%s\n", channel, message);
  else
    fprintf(our_conn, "DO PART %s\n", channel);
}

void IRC_CTCP(const char *target, const char *command, const char *data)
{
  if (data!=NULL)
    fprintf(our_conn, "DO CTCP %s %s %s\n", target, command, data);
  else
    fprintf(our_conn, "DO CTCP %s %s\n", target, command);
}

void IRC_CTCPReply(const char *target, const char *command, const char *data)
{
  IRC_Notice(target, Util_CTCPFmtD(command, data));
}

const char * IRC_VersionInfo()
{
  return "Lachesis Message Interface 0.1";
}

void IRC_Nick(const char *nick)
{
  fprintf(our_conn, "DO NICK %s\n", nick);
}

void IRC_Public(const char *channel, const char *message)
{
  fprintf(our_conn, "DO MSG %s %s\n", channel, message);
}

void IRC_Action(const char *channel, const char *message)
{
  fprintf(our_conn, "DO DESCRIBE %s %s\n", channel, message);
}

void IRC_MSG(const char *target, const char *message)
{
  fprintf(our_conn, "DO MSG %s %s\n", target, message);
}

void IRC_Describe(const char *target, const char *message)
{
  fprintf(our_conn, "DO DESCRIBE %s %s\n", target, message);
}

void IRC_Wallop(const char *message)
{
  fprintf(our_conn, "DO WALLOP %s\n", message);
}

void IRC_Join(const char *channel, const char *key)
{
  if (key!=NULL)
    fprintf(our_conn, "DO JOIN %s %s\n", channel, key);
  else
    fprintf(our_conn, "DO JOIN %s\n", channel);
}

void IRC_Quit(const char *message)
{
  fprintf(our_conn, "DO QUIT %s\n", message);
}

TF IRC_IsConnected()
{
  return !!our_conn;
}

const char * IRC_GetNick()
{
  return our_nick;
}

const char * IRC_GetHost()
{
  return our_host;
}

const char * IRC_HostmaskNick(const char *prefix)
{
  if (prefix!=internal_buffer)
    Util_Copy(internal_buffer, prefix, ALLOC_MINIBUFFER);
  else
    if (strchr(internal_buffer, '!')!=NULL)
      PrefixSeparate();
  return internal_buffer;
}

const char * IRC_HostmaskUser(const char *prefix)
{
  const char *temp;
  if (prefix!=internal_buffer)
    Util_Copy(internal_buffer, prefix, ALLOC_MINIBUFFER);
  else
    if (strchr(internal_buffer, '!')!=NULL)
      PrefixSeparate();
  temp = strchr(internal_buffer, 0);
  return temp+1;
}

const char * IRC_HostmaskHost(const char *prefix)
{
  const char *temp;
  if (prefix!=internal_buffer)
    Util_Copy(internal_buffer, prefix, ALLOC_MINIBUFFER);
  else
    if (strchr(internal_buffer, '!')!=NULL)
      PrefixSeparate();
  temp = strchr(internal_buffer, 0);
  temp = strchr(temp+1, 0);
  return temp+1;
}

void IRC_DCCChatSend(const char *target)
{
  fprintf(our_conn, "DO DCC CHAT %s\n", target);
}

void IRC_DCCChatAccept(void *cookie)
{
  fprintf(our_conn, "DO DCC CHAT %s\n", ((PDCC)cookie)->nick);
}

void IRC_DCCChatReject(void *cookie)
{
  fprintf(our_conn, "DO DCC CLOSE CHAT %s\n", ((PDCC)cookie)->nick);
  RemDCC((PDCC)cookie);
}

void IRC_DCCChatMessage(void *cookie, const char *message)
{
  fprintf(our_conn, "DO DMSG %s %s\n", ((PDCC)cookie)->nick, message);
}

void IRC_DCCChatClose(void *cookie)
{
  fprintf(our_conn, "DO DCC CLOSE CHAT %s\n", ((PDCC)cookie)->nick);
  RemDCC((PDCC)cookie);
}

const char * IRC_DCCGetNick(void *cookie)
{
  return ((PDCC)cookie)->nick;
}

void IRC_DCCSend(const char *target, const char *fn)
{
  fprintf(our_conn, "DO DCC SEND %s %s\n", target, fn);
}

void OnDCCChatRequest(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  if (GetDCC(s1)!=NULL)
    // This is a problem with ircII; all we can do is ignore here for now.
    return ;
  NewDCC(s1);
  s2 = str;
  PrefixGenerate(&s2);
  HandleDCCChatRequestPfx((void*)(dcc_list), internal_buffer);
}

void OnDCCChatOpened(const char *str)
{
  PDCC temp;
  temp = GetDCC(str);
  if (temp==NULL)
    fputs("MsgIrc: Missing DCC cookie.", stderr);
  else
    HandleDCCChatOpened((void*)(temp));
}

void OnDCCChatClosed(const char *str)
{
  PDCC temp;
  temp = GetDCC(str);
  if (temp==NULL)
    fputs("MsgIrc: Missing DCC cookie.", stderr);
  else {
    HandleDCCChatClosed((void*)(temp));
    RemDCC(temp);
  }
}

void OnDCCChatMSG(const char *str)
{
  const char *s1, *s2;
  PDCC temp;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  temp = GetDCC(s1);
  if (temp==NULL)
    fputs("MsgIrc: Missing DCC cookie.", stderr);
  else
    HandleDCCChatMessage((void*)(temp), s2);
}

void OnNumeric(const char *str)
{
  char *temp;
  long int num;
  num = strtol(str, &temp, 10);
  HandleNumeric(num, temp);
}

void OnNotice(const char *str)
{
  const char *temp;
  temp = str;
  PrefixGenerate(&temp);
  HandleNotice(internal_buffer, our_nick, temp);
}

void OnCTCP(const char *str)
{
  const char *s1, *s2, *temp;
  char nick[ALLOC_NICK];
  Util_Copy(nick, str, ALLOC_NICK);
  *(strchr(nick, 32)) = 0;
  s2 = str;
  PrefixGenerate(&s2);
  s1 = Util_Separate(&s2, 32);
  if ((temp = HandleCTCP(internal_buffer, s1, s2))!=NULL)
    IRC_CTCPReply(nick, s1, temp);
}

void OnCTCPReply(const char *str)
{
  const char *temp;
  PrefixGenerate(&temp);
  HandleCTCPReply(internal_buffer, temp);
}

void OnNames(const char *str)
{
  // 353 involves a number of names with a trailing space.
  int c;
  const char **ptr;

  c = Util_Count(str, 32);
  ptr = (const char **)malloc((c+1)*sizeof(char *));
  Util_Explode(str, 32, ptr, c+1);
  HandleNames(ptr[0], ptr+1, c-1);
  free(ptr);
}

void OnInvite(const char *str)
{
  const char *temp;
  temp = str;
  PrefixGenerate(&temp);
  HandleInvite(internal_buffer, temp);
}

void OnKick(const char *str)
{
  const char *temp[4];
  Util_Explode(str, 32, temp, 4);
  HandleKick(temp[0], temp[1], temp[2], temp[3]);
}

void OnMSG(const char *str)
{
  const char *temp;
  temp = str;
  PrefixGenerate(&temp);
  HandleMSG(internal_buffer, temp);
}

void OnPublic(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  PrefixGenerate(&s2);
  s1 = Util_Separate(&s2, 32);
  HandlePublic(internal_buffer, s1, s2);
}

void OnAction(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  PrefixGenerate(&s2);
  s1 = Util_Separate(&s2, 32);
  HandleCTCP(internal_buffer, s1, Util_Format("%s %s", "ACTION", s2, NULL));
}

void OnJoin(const char *str)
{
  char *temp;
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  if (strcasecmp(our_nick, s1)==0) {
    // Skip the user@host.
    s2 = strchr(s2, 32);
    s2++;
    HandleJoined(s2);
  } else {
    s2 = str;
    PrefixGenerate(&s2);
    HandleJoin(internal_buffer, s2);
  }
}

void OnNewNick(const char *str)
{
  HandleOurNickChanged(str);
  Util_Copy(our_nick, str, ALLOC_NICK);
}

void OnNick(const char *str)
{
  const char *s1, *s2;
  s2 = str;
  PrefixGenerate(&s2);
  HandleNickChange(internal_buffer, s2);
}

void OnPart(const char *str)
{
  char nick[ALLOC_NICK];
  const char *s1, *s2;
  s2 = str;
  s1 = Util_Separate(&s2, 32);
  Util_Copy(nick, s1, ALLOC_NICK);
  s1 = Util_Separate(&s2, 32);
  if (s1 == NULL)
    HandlePart(nick, s2, NULL);
  else
    HandlePart(nick, s1, s2);
}

void OnQuit(const char *str)
{
  const char *nick, *msg;
  msg = str;
  nick = Util_Separate(&msg, 32);
  if (strcasecmp(nick, our_nick)==0)
    return ;
  HandleQuit(nick, msg);
}

void OnDisconnect(const char *message)
{
  HandleDisconnect(message);
}

void OnConnect(const char *str)
{
  // Placeholder
}

void OnUserhost(const char *str)
{
  // This is simpler than handling a 302, and less prone to unforseen problems
  const char *ptr[3];
  Util_Explode(str, 32, ptr, 3);
  HandleUserhost(ptr[0], ptr[1], ptr[2]);
}

struct strtofunc_map {
  const char *str;
  void (*fun)(const char *);
};

const struct strtofunc_map EVENT_LIST[] = {
  { "DCCCHATREQ", &OnDCCChatRequest },
  { "DCCCHATOPEN", &OnDCCChatOpened },
  { "DCCCHATCLOSE", &OnDCCChatClosed },
  { "DCCMSG", &OnDCCChatMSG },
  { "CONNECT", &OnConnect },
  { "NUMERIC", &OnNumeric },
  { "NOTICE", &OnNotice },
  { "CTCP", &OnCTCP },
  { "CTCPREPLY", &OnCTCPReply },
  { "NAMES", &OnNames },
  { "INVITE", &OnInvite },
  { "KICK", &OnKick },
  { "MSG", &OnMSG },
  { "PUBLIC", &OnPublic },
  { "ACTION", &OnAction },
  { "JOIN", &OnJoin },
  { "NICK", &OnNick },
  { "PART", &OnPart },
  { "QUIT", &OnQuit },
  { "DISCONNECT", &OnDisconnect },
  { "NEWNICK", &OnNewNick },
  { "USERHOST", &OnUserhost }, 
  { NULL, NULL }
};

void StartMessage()
{
  char *nick = NULL, *input = NULL, *temp;
  size_t len = 0;
  fputs("SENDNICK\n", our_conn);
  getline(&nick, &len, our_conn);
  *(strchr(nick, '\n'))=0;
  fputs("SENDNAME\n", our_conn);
  getline(&input, &len, our_conn);
  *(strchr(input, '\n'))=0;
  Util_Copy(our_name, input, ALLOC_NAME);
  fputs("SENDUH\n", our_conn);
  getline(&input, &len, our_conn);
  *(strchr(input, '\n'))=0;
  temp = strchr(input, '@');
  *temp = 0;
  temp++;
  Util_Copy(our_user, input, ALLOC_USER);
  Util_Copy(our_host, temp, ALLOC_HOST);
  if (strcasecmp(our_nick, nick)!=0)
    fprintf(our_conn, "DO NICK %s", our_nick);
  ConnectionReady();
  while (1) {
    const char *cmd, *data;
    int c;

    if (our_conn==NULL)
      return;

    if (getline(&input, &len, our_conn)<0)
      if (feof(our_conn)) {
	fclose(our_conn);
	our_conn = NULL;
	free (input);
	HandleDisconnect("No message available.");
	return ;
      } else
	fputs("MsgIrc: Unknown error reading socket.", stderr);
    *(strchr(input, '\n'))=0;
    data = (const char *)(input);
    cmd = Util_Separate(&data, 32);
    if (cmd==NULL) {
      cmd = data;
      data = NULL;
    }
    for (c=0;EVENT_LIST[c].str!=NULL;c++)
      if (strcasecmp(cmd, EVENT_LIST[c].str)==0) {
	(*(EVENT_LIST[c].fun))(data);
	break;
      }
  }
}

void IRC_Connect(const char *server, const int port, const char *nick,
		 const char *user, const char *s_name)
{
  int sock;
  struct sockaddr_in name;

  Util_Copy(our_nick, nick, ALLOC_NICK);
  /* Util_Copy(our_user, user, ALLOC_USER); */
  /* Util_Copy(our_name, s_name, ALLOC_NAME); */

  /* Wouldn't it be nice?  Heh.  For this interface, we have to take what
   * the negotiating client gives us for user, name, and server.
   * We can assert the nick, at least.
   */

  sock = socket(PF_INET, SOCK_STREAM, 0);
  if (sock < 0) {
    fputs("MsgIrc: Unable to allocate a socket!", stderr);
    exit (1);
  }

  name.sin_family = AF_INET;
  if (options.port)
    name.sin_port = htons(options.port);
  else
    name.sin_port = htons(5013); // Different default for this interface
  name.sin_addr.s_addr = htonl (INADDR_LOOPBACK);
  if (bind(sock, (struct sockaddr *)(&name), sizeof(name)) < 0) {
    fputs("MsgIrc: Unable to bind socket!", stderr);
    exit (1);
  }

  if (listen(sock, 1) < 0) {
    fputs("MsgIrc: Unable to listen on socket!", stderr);
    exit (1);
  }

  while (1) {
    struct sockaddr_in addr;
    socklen_t len = sizeof(struct sockaddr_in);
    int newconn;
    char *buf = NULL;
    size_t buflen = 0;
    if ((newconn=accept(sock, (struct sockaddr *)(&addr), &len))<0) {
      fputs("MsgIrc: Unable to accept connections!", stderr);
      exit (1);
    }
    our_conn = fdopen(newconn, "a+");
    if (getline(&buf, &buflen, our_conn)==-1) {
      if (feof(our_conn)) {
	fclose(our_conn);
	continue;
      } else {
	fputs("MsgIrc: Unknown error reading socket input!", stderr);
	exit (1);
      }
    }

    if (strcmp(buf, "LACHESIS MESSAGE INTERFACE\n")==0) {
      close (sock);
      StartMessage();
      return ;
    }
    fclose(our_conn);
  }
}

void LowIrcCleanup()
{
}
